<?php
session_start();

// DB connection
include(__DIR__ . '/../includes/db.php');

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: ../user/userlogin.php");
    exit;
}

$user_id = (int)$_SESSION['user_id'];

// Fetch user data
$stmt = $conn->prepare("SELECT fullname, email, password FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

$user_name = $user['fullname'] ?? 'User';
$user_email = $user['email'] ?? '';
$user_password = $user['password'] ?? '';

// Default profile image
$profile_image = "../assets/image/starserver.png";

// Fetch user's active hosting plans
$stmt2 = $conn->prepare("
    SELECT plan_name, domain_name, activated_at, expiry_date, payment_status
    FROM hosting_plans
    WHERE user_id = ? AND payment_status='approved'
    ORDER BY expiry_date ASC
");
$stmt2->bind_param("i", $user_id);
$stmt2->execute();
$plans_result = $stmt2->get_result();

// Calculate stats
$active_plans_count = $plans_result->num_rows;
$next_renewal_date = null;
$days_remaining = 0;

if($active_plans_count > 0){
    $today_ts = strtotime(date('Y-m-d'));
    $earliest_expiry_ts = null;

    while($plan = $plans_result->fetch_assoc()){
        $expiry_ts = strtotime($plan['expiry_date']);
        if(!$earliest_expiry_ts || $expiry_ts < $earliest_expiry_ts){
            $earliest_expiry_ts = $expiry_ts;
        }
    }

    $next_renewal_date = date('d M Y', $earliest_expiry_ts);
    $days_remaining = max(0, floor(($earliest_expiry_ts - $today_ts)/86400));
}

// Account status
$account_status = ($active_plans_count > 0 && $days_remaining > 0) ? 'Active' : 'Inactive';

include(__DIR__ . '/../includes/header.php');
include(__DIR__ . '/../includes/navbar.php');
?>

<main class="content container my-5">

    <!-- Top Bar -->
    <div class="topbar mb-4 d-flex justify-content-between align-items-center">
        <h4>Star Server – <span>User Dashboard</span></h4>
        <div class="user_profile text-end">
            <img src="<?= $profile_image; ?>" alt="User Profile" class="rounded-circle" width="50">
            <h5 class="mt-1"><?= htmlspecialchars($user_name); ?></h5>
        </div>
    </div>

    <!-- Dashboard Stats -->
    <section id="dashboard" class="section active-section">
        <div class="row g-4 mb-4">

            <div class="col-md-3">
                <div class="card stat-card blue shadow-sm">
                    <div class="card-body text-center">
                        <h6 class="text-white">Active Hosting Plans</h6>
                        <h2 class="fw-bold"><?= $active_plans_count; ?></h2>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stat-card green shadow-sm">
                    <div class="card-body text-center">
                        <h6 class="text-white">Account Status</h6>
                        <h2 class="fw-bold"><?= $account_status; ?></h2>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stat-card orange shadow-sm">
                    <div class="card-body text-center">
                        <h6 class="text-white">Next Renewal Date</h6>
                        <h2 class="fw-bold"><?= $next_renewal_date ?? 'N/A'; ?></h2>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stat-card pink shadow-sm">
                    <div class="card-body text-center">
                        <h6 class="text-white">Days Remaining</h6>
                        <h2 class="fw-bold"><?= $days_remaining; ?></h2>
                    </div>
                </div>
            </div>

        </div>

        <!-- User Profile Info -->
        <div class="card mb-5 shadow-sm">
            <div class="card-header bg-secondary text-white">
                <h5 class="mb-0">Profile Information</h5>
            </div>
            <div class="card-body">
                <form id="profileForm" method="POST" action="update_profile.php">
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">Full Name</label>
                            <input type="text" name="fullname" class="form-control" value="<?= htmlspecialchars($user_name); ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Email</label>
                            <input type="email" name="email" class="form-control" value="<?= htmlspecialchars($user_email); ?>" required>
                        </div>
                        <div class="col-md-4">
        <label class="form-label">Password</label>
        <input type="text" class="form-control"
               value="<?= htmlspecialchars($user['password']) ?>" readonly>
    </div>
                    </div>
                    <div class="mt-3 text-end">
                        <button type="submit" class="btn btn-primary">Update Profile</button>
                    </div>
                </form>
            </div>
        </div>

        <!-- User Plans Table -->
        <?php if($active_plans_count > 0): ?>
        <div class="card shadow-sm">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">Your Hosting Plans</h5>
            </div>
            <div class="card-body p-0">
                <table class="table table-striped table-hover mb-0 text-center responsive-table">
                    <thead class="table-dark">
                        <tr>
                            <th>#</th>
                            <th>Plan Name</th>
                            <th>Domain</th>
                            <th>Activated At</th>
                            <th>Expiry Date</th>
                            <th>Days Left</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $stmt2->execute();
                        $plans_result = $stmt2->get_result();
                        $i = 1;
                        while($plan = $plans_result->fetch_assoc()):
                            $expiry_ts = strtotime($plan['expiry_date']);
                            $today_ts  = strtotime(date('Y-m-d'));
                            $days_left = max(0, floor(($expiry_ts - $today_ts)/86400));
                            $status_badge = ($days_left > 0) ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-danger">Expired</span>';
                        ?>
                        <tr>
                            <td><?= $i++; ?></td>
                            <td><?= htmlspecialchars($plan['plan_name']); ?></td>
                            <td><?= htmlspecialchars($plan['domain_name']); ?></td>
                            <td><?= date('d M Y', strtotime($plan['activated_at'])); ?></td>
                            <td><?= date('d M Y', $expiry_ts); ?></td>
                            <td><?= $days_left; ?></td>
                            <td><?= $status_badge; ?></td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php else: ?>
        <div class="text-center mt-5 text-muted">
            <i class="bi bi-hdd-network fs-1 mb-2"></i>
            <p>No active hosting plans found.</p>
        </div>
        <?php endif; ?>

    </section>
</main>

<?php
include(__DIR__ . '/../includes/footer.php');
include(__DIR__ . '/../includes/scripts.php');
?>
<style>
    @media (max-width:675px) {
    .topbar h4 {
    margin: 0;
    font-weight: 600;
    font-size: 23px !important;
    color: #000000;
}
}

/* ===============================
   Responsive Table (Mobile View)
================================ */
@media (max-width: 768px) {

    /* Hide table head */
    .responsive-table thead {
        display: none;
    }

    .responsive-table,
    .responsive-table tbody,
    .responsive-table tr,
    .responsive-table td {
        display: block;
        width: 100%;
    }

    /* Card container */
    .responsive-table tr {
        background: linear-gradient(180deg, #ffffff, #fafafa);
        margin-bottom: 18px;
        padding: 14px 16px;
        border-radius: 16px;
        border-left: 5px solid #4e73df;
        box-shadow: 0 10px 25px rgba(0,0,0,0.08);
        transition: transform 0.25s ease, box-shadow 0.25s ease;
    }

    /* Soft tap effect */
    .responsive-table tr:active {
        transform: scale(0.985);
        box-shadow: 0 6px 16px rgba(0,0,0,0.12);
    }

    /* Row items */
    .responsive-table td {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 8px 0;
        border: none;
        font-size: 15px;
        line-height: 1.4;
        white-space: normal;
    }

    /* Label */
    .responsive-table td::before {
        content: attr(data-label);
        font-weight: 600;
        color: #6c757d;
        font-size: 14px;
    }

    /* Value emphasis */
    .responsive-table td:not(:last-child) {
        border-bottom: 1px dashed #e6e6e6;
    }

    /* Domain highlight */
    .responsive-table td[data-label="Domain"] {
        font-weight: 600;
        color: #1a1a1a;
        font-size: 16px;
    }

    /* Badge styling */
    .responsive-table td span.badge {
        font-size: 13px;
        padding: 6px 14px;
        border-radius: 20px;
        font-weight: 600;
    }

    /* Status colors */
    .badge.bg-success {
        background: linear-gradient(135deg, #1cc88a, #0f9d58) !important;
    }

    .badge.bg-danger {
        background: linear-gradient(135deg, #e74a3b, #c0392b) !important;
    }
}


</style>


